# Module Tests
import asyncio
import time
import sys
import os
import ast

from py_pli.pylib import VUnits, GlobalVar
from predefined_tasks.common.helper import send_to_gc, report_path, write_header_to_report

from fleming.common.firmware_util import get_fan_control_endpoint
from fleming.common.firmware_util import get_node_endpoint
from fleming.common.firmware_util import get_serial_endpoint
from fleming.common.firmware_util import get_corexy_mover_endpoint
from fleming.common.firmware_util import get_barcode_reader_endpoint
from virtualunits.vu_scan_table import BarcodeReaderIDs
import urpc_enum.barcodereaderparameter as bcr_param
from urpc_enum.barcodereaderparameter import BarcodeReaderParameter
import config_enum.scan_table_enum as scan_table_enum


# Hardware IO Mapping

## Fans

fans = {
    'base1' : {'node': 'fmb_fan','channel': 2}, # FAN 3 
    'base2' : {'node': 'fmb_fan','channel': 3}, # FAN 4 
    'base3' : {'node': 'fmb_fan','channel': 4}, # FAN 5 
    'int'   : {'node': 'fmb_fan','channel': 0}, # FAN 1
    'ext'   : {'node': 'fmb_fan','channel': 1}, # FAN 2
    'pc'    : {'node': 'fmb_fan','channel': 6}, # FAN 7 (FMB Rear side)
    'fl'    : {'node': 'eef_fan','channel': 0}, # FAN 0 (connected to EEF)
    'trf'   : {'node': 'fmb_fan','channel': 7}, # FAN SW
}

## Temperature Sensors

temp_sensors = { # tmb = temp.-measurement-board
    'tmb_upper' : {'node': 'fmb', 'number': 0, 'conv': 256}, # Measurement chamber ceiling
    'tmb_lower' : {'node': 'fmb', 'number': 1, 'conv': 256}, # Measurement chamber bottom
    'tmb_in'    : {'node': 'fmb', 'number': 2, 'conv': 256}, # Airventilation unit inlet to internal fan
    'tmb_out'   : {'node': 'fmb', 'number': 3, 'conv': 256}, # Airventilation unit outlet
    'tmb_fl'    : {'node': 'fmb', 'number': 5, 'conv': 256}, # Flashlamp Fan
    'tmb_int'   : {'node': 'fmb', 'number': 6, 'conv': 256}, # AirV. Peltier Cold side
    'tmb_ext'   : {'node': 'fmb', 'number': 7, 'conv': 256}, # AirV. Peltier Hot side
    'tmb_am'    : {'node': 'fmb', 'number': 16, 'conv': 175}, # Ambient Temperature Sensor
    'tmb_hum'   : {'node': 'fmb', 'number': 17, 'conv': 100}, # Humidity Sensor
    'tmb_p1'    : {'node': 'eef', 'number': 15, 'conv': 256},   # PMT 1 Temp
    'tmb_p2'    : {'node': 'eef', 'number': 16, 'conv': 256},  # PMT 2 Temp
}

## Temperature-Elements

temp_elements = {
    'htu'   : 0, # Upper heating
    'htl'   : 1, # Lower heating
    'htb'   : 2, # Dispenser bottle heating
    'tec'   : 3, # 
    'av1'   : 15, # AVU Peltier Pin 1
    'av2'   : 16, # AVU Peltier Pin 2
    'pmt_1' : 9, # PMT 1 Peltier
    'pmt_2' : 10, # PMT 1 Peltier
}

## Chip warnings
pwm_status = {
    'AV_otw'   : 11, # PWM 2 Driver (AV1+ 2) Overtemperature Warning
    'AV_fault' : 10, # PWM 2 Driver (AV1+ 2) Fault
    'HT_otw'   : 9, # PWM 1 Driver (HTL+ U) Overtemperature Warning
    'HT_fault' : 8, # PWM 1 Driver (HTL+ U) Fault
    'PMT_AL_otw'  : 25, # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC overtemperature warning (active low)
    'PMT_AL_fault': 24 # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC fault signal (active low)
}

# Node endpoints

fmb = get_node_endpoint('fmb') # Access of low level board IO
eef = get_node_endpoint('eef') # Access of low level board IO
scan_table = VUnits.instance.hal.scan_table
#bcr_back_endpoint = get_barcode_reader_endpoint('bcr3')
#bcr_right_endpoint = get_barcode_reader_endpoint('bcr4')
#bcr_front_endpoint = get_barcode_reader_endpoint('bcr5')


SCRIPT_NAME = os.path.splitext(__file__)[0]  

# Helper Functions

def create_function_list(filename, prefix =""):
    #
    with open(filename, 'r') as file:
        tree = ast.parse(file.read(), filename=filename)

    function_names = [node.name for node in ast.walk(tree) if isinstance(node, ast.FunctionDef)]   
    list_name = filename.replace(".py", ".fun")

    with open(list_name, 'w') as file:
        for f in function_names:
            file.write(prefix + f + '\n')
    
    return(list_name)


# High level Module Tests 
async def barcode_reader_option_test(module_no = '0'):
    '''
    Module Test for Barcode Reader Option HH36660001
    Checks barcodereader of the bcr option (2-4), configures it and trigger it

    BCR Option connected to EEF-Board: BCR EXT
    
    Returns: Report Filename
    '''

    TEST_NAME = sys._getframe().f_code.co_name
    MODULE_NAME = 'HH36660001'

    report_file = report_path(f'{MODULE_NAME}_SN{module_no}_{TEST_NAME}')
    
    bcr_option_list =[BarcodeReaderIDs.BCRBack, BarcodeReaderIDs.BCRRight, BarcodeReaderIDs.BCRFront]

    msg = f'Test {TEST_NAME} started... '
    await send_to_gc(msg, False, False)

    msg = 'Barcodereader Front, Right, Rear initializing...'
    await send_to_gc(msg, False, False)

    GlobalVar.set_stop_gc(False)
    with open(report_file, "w") as report:
        write_header_to_report(report)
        msg = f'{TEST_NAME} Result: '
        await send_to_gc(msg, False, False, report)
        for bcr in bcr_option_list:
            if GlobalVar.get_stop_gc():
                await send_to_gc(f"Script stopped by user", True, True, report)
                return (report_file)
            if (await scan_table.is_bcr_available(bcr)) == None:
                read_result = await scan_table.ReadBarcode(bcr)
                msg = f'Barcode Reader {bcr.name}: OK, BC: {read_result}'  
                await send_to_gc(msg, False, False, report)

            else:
                msg = f'Barcode Reader {bcr.name}: FAILED' 
                await send_to_gc(msg, True, True, report) 
    
    return(report_file)
    
async def temperature_control_unit_test(power = 96, limit = -2.0):
async def temperature_control_unit_test(module_no = '0', power = 96, limit = -2.0):
    '''
    Module Test for Temperature Control Unit HH36090000
    starts TEC for 60 secs and detects temperature gradient
    
    TEC connected to FMB Board: AV1, AV2
    TMB INT connected to FMB-Board: 6
    TMB EXT connected to FMB-Board: 7
    Small Radial Fan connected to FMB-Board: FAN1
    Big Axial Fan connected to FMB-Board: FAN2

    Returns report file name
    '''
    DURATION = 60
    SAMPLE_RATE = 1
    T_MAX = 60
    T_SCALE = 30
    
    TEST_NAME = sys._getframe().f_code.co_name
    MODULE_NAME = 'HH36090000'
    DURATION = 60
    SAMPLE_RATE = 1
    T_MAX = 80
    T_SCALE = 30
    limit_f = float(limit) if limit else -2.0

    report_file = report_path(f'{MODULE_NAME}_SN{module_no}_{TEST_NAME}')

    bar = lambda x, x_max, width = T_SCALE, char = "|" : char.ljust(int(x / x_max * width), char)
    t_cool_1 = await _get_tmp('tmb_int')
    t_hot_1 = await _get_tmp('tmb_ext')
    msg = await _tcu_on(power)
    await send_to_gc(msg, False, False)
    
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    with open(report_file, "w") as report:
        write_header_to_report(report)     
        while (time.time() - start_on) < DURATION:
            if GlobalVar.get_stop_gc():
                await _tcu_off()
                await send_to_gc(f"Script stopped by user", True, True, report)
                return "Stopped by user"
            t_cool = await _get_tmp('tmb_int')
            t_warn = await _get_tmp('tmb_ext')
            if t_warn > T_MAX:
                await _tcu_off()
                await send_to_gc(f"Script stopped: TEC Over-Temperature", True, True, report)
                return "Stopped by system "

            result=bar(t_cool, T_SCALE)
            msg = f"{t_cool:.2f}deg C \t {result}"
            await send_to_gc(msg, False, False)     
            await asyncio.sleep(SAMPLE_RATE)
        msg = await _tcu_off()
        d_t_cool = await _get_tmp('tmb_int') - t_cool_1
        d_t_hot = await _get_tmp('tmb_ext') - t_hot_1
        if d_t_cool > limit_f:
            msg = f'{TEST_NAME} Result FAILED, Result: AVU Cooling Rate: {d_t_cool:.2f}K/min, TEC_Hot: {d_t_hot:.2f}K/min'  
            await send_to_gc(msg, False, False, report)
        else:
            msg = f'{TEST_NAME} Result OK, Result: AVU Cooling Rate: {d_t_cool:.2f}K/min, TEC_Hot: {d_t_hot:.2f}K/min' 
            await send_to_gc(msg, False, False, report)
        
    return(report_file) 

async def pmt_cooling_test(module_no = '0', power_percent = 96, limit = -1.0):
    '''
    Module Test for cooling function of PMT Unit HH36700140, 60
    TEC Connected to EEF-Board: PMT-TEC1
    TMB connected to EEF-Board: PMT1-SENS

    Returns report file name
    '''    
    TEST_NAME = sys._getframe().f_code.co_name
    MODULE_NAME = 'HH36700140'
    DURATION = 60
    SAMPLE_RATE = 1
    T_SCALE = 30

    limit_f = float(limit) if limit else -2.0

    report_file = report_path(f'{MODULE_NAME}_SN{module_no}_{TEST_NAME}')

    bar = lambda x, x_max, width = T_SCALE, char = "|" : char.ljust(int(x / x_max * width), char)
    t_cool_1 = await _get_tmp('tmb_p1')
    msg = await _pmt_tec(power_percent)
    await send_to_gc(msg, False, False)
    
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    with open(report_file, "w") as report:
        write_header_to_report(report)     
        while (time.time() - start_on) < DURATION:
            if GlobalVar.get_stop_gc():
                await _pmt_tec(0)
                await send_to_gc(f"Script stopped by user", True, True, report)
                return "Stopped by user"
            t_cool = await _get_tmp('tmb_p1')
            result=bar(t_cool, T_SCALE)
            msg = f"{t_cool:.2f}deg C \t {result}"
            await send_to_gc(msg, False, False)     
            await asyncio.sleep(SAMPLE_RATE)
        msg = await _pmt_tec(0)
        d_t_cool = await _get_tmp('tmb_p1') - t_cool_1
        if d_t_cool > limit_f:
            msg = f'{TEST_NAME}: Result FAILED: PMT Cooling Rate: {d_t_cool:.2f}K/min'  
            await send_to_gc(msg, False, False, report)
        else:
            msg = f'{TEST_NAME}: Result OK: PMT Cooling Rate: {d_t_cool:.2f}K/min' 
            await send_to_gc(msg, False, False, report)
        
    return(report_file)    

async def scantable_heater_test(module_no = '0', power_percent = 96, limit = 2.0):
    '''
    Module Test for heating function of scantable unit HH36030001

    Heating foil connected to FMB-Board: HTL
    Temperature sensor connected to FMB-Board: 1

    Returns report file name
    '''
    TEST_NAME = sys._getframe().f_code.co_name
    MODULE_NAME = 'HH36030001'
    DURATION = 60
    SAMPLE_RATE = 1
    T_MAX = 60
    T_SCALE = 30

    report_file = report_path(f'{MODULE_NAME}_SN{module_no}_{TEST_NAME}')

    bar = lambda x, x_max, width = T_SCALE, char = "|" : char.ljust(int(x / x_max * width), char)

    msg = f'Test {TEST_NAME} started... '
    await send_to_gc(msg, False, False)

    msg = 'Switch on heater... '
    await send_to_gc(msg, False, False)

    msg = await _scantable_heater(power_percent)
    await send_to_gc(msg, False, False)

    t_heat1 = await _get_tmp('tmb_lower')
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    with open(report_file, "w") as report:
        write_header_to_report(report)
        while (time.time() - start_on) < DURATION:
            if GlobalVar.get_stop_gc():
                await _scantable_heater(0)
                await send_to_gc(f"Script stopped by user", True, True, report)
                return "Stopped by user"

            t_heat = await _get_tmp('tmb_lower')
            if t_heat > T_MAX:
                await _scantable_heater(0)
                await send_to_gc(f"!!!ERROR: Temperature above {T_MAX}!!!", True, True, report)
                return "Stopped by system"

            result = bar(t_heat, T_MAX)
            msg = f"{t_heat:.2f}deg C \t {result}"
            await send_to_gc(msg, False, False)     
            await asyncio.sleep(SAMPLE_RATE)

        msg = await _scantable_heater(0)
        await send_to_gc(msg, False, False)
        d_t_heat = await _get_tmp('tmb_lower') - t_heat1
        
        if d_t_heat > limit:
            msg = f'{TEST_NAME} Result OK: Heating Rate Scantable: {d_t_heat:.2f}K/min'  
            await send_to_gc(msg, False, False, report)
        else:
            msg = f'{TEST_NAME} Result FAILED: Heating Rate Scantable: {d_t_heat:.2f}K/min'  
            await send_to_gc(msg, False, False, report)

    return(report_file)

async def barcode_reader_option_test(module_no = '0'):
    '''
    Module Test for Barcode Reader Option HH36660001
    Checks barcodereader of the bcr option (2-4), configures it and trigger it

    BCR Option connected to EEF-Board: BCR EXT
    
    Returns: Report Filename
    '''

    TEST_NAME = sys._getframe().f_code.co_name
    MODULE_NAME = 'HH36660001'

    report_file = report_path(f'{MODULE_NAME}_SN{module_no}_{TEST_NAME}')
    
    bcr_option_list =[BarcodeReaderIDs.BCRBack, BarcodeReaderIDs.BCRRight, BarcodeReaderIDs.BCRFront]

    msg = f'Test {TEST_NAME} started... '
    await send_to_gc(msg, False, False)

    msg = 'Barcodereader Front, Right, Rear initializing...'
    await send_to_gc(msg, False, False)

    GlobalVar.set_stop_gc(False)
    with open(report_file, "w") as report:
        write_header_to_report(report)
        msg = f'{TEST_NAME} Result: '
        await send_to_gc(msg, False, False, report)
        for bcr in bcr_option_list:
            if GlobalVar.get_stop_gc():
                await send_to_gc(f"Script stopped by user", True, True, report)
                return (report_file)
            if (await scan_table.is_bcr_available(bcr)) == None:
                read_result = await scan_table.ReadBarcode(bcr)
                msg = f'Barcode Reader {bcr.name}: OK, BC: {read_result}'  
                await send_to_gc(msg, False, False, report)

            else:
                msg = f'Barcode Reader {bcr.name}: FAILED' 
                await send_to_gc(msg, True, True, report) 
    
    return(report_file)



# Low Level Functions

async def _fan_pwm(fan_name, pwr):
    """
    Internal function to switch the fan to pwr (percentage of maximum rotation speed)
    fan_name: base1, base2, base3, int, ext, pc, fl, trf
    pwr: 0...100%

    Returns status message
    """
    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    fan = get_fan_control_endpoint(fans[fan_name]['node'])
    channel = fans[fan_name]['channel']
    await fan.SetSpeed(channel, pwr, timeout=1)
    await fan.Enable(channel, (pwr > 0), timeout=1)
    return f"Fan '{fan_name}' @ {int(pwr)}%"

# Temp.-Sensors

# [] tested 
async def _get_tmp(tmb_name):
    """
    Intenal function to read the temperature value of the according sensor
    Needs the dict of the sensors

    tmb_name:
    tmb_upper: Upper Heating
    tmb_lower: Lower Heating
    tmb_in:    AVU internal Fan inlet 
    tmb_out:   AVU internal Fan outlet
    tmb_fl:    Flashlamp Temp
    tmb_int:   AVU Internal Fan (Cool side Peltier) 
    tmb_ext:   AVU external Fan (Hot side Peltier)
    tmb_am:    Ambient Temp
    tmb_hum:   Humidity Sensor
    tmb_p1:    PMT 1  
    tmb_p2:    PMT 2
    
    Returns Temperature Value in Degrees Celsius
    """
    if tmb_name not in temp_sensors:
        raise ValueError(f"{tmb_name} not valid.")
    
    node = get_node_endpoint(temp_sensors[tmb_name]['node'])
    value = (await node.GetAnalogInput(temp_sensors[tmb_name]['number']))[0] * int(temp_sensors[tmb_name]['conv'])
    return value

async def _tcu_on(pwm_percent = 65, fan = 10):
    '''
    Internal function to switch TCU TEC Element to pwm_percent
    Returns status message
    '''    
    min_pwm = 0.05
    pwm = pwm_percent / 100
    fmb = get_node_endpoint('fmb')
    # fmb = VUnits.instance.hal.nodes['Mainboard']
    await _fan_pwm('int', fan)
    await _fan_pwm('ext', 100)
    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

    await fmb.SetAnalogOutput(p_1, (min_pwm))
    await fmb.SetAnalogOutput(p_2, (pwm))

    return('AVU TE switched on')

# Turns Peltiers of airventilation unit OFF
# [] tested     
async def _tcu_off():
    '''
    Internal function to switch TCU TEC Element off
    Returns status message
    '''
    fmb = get_node_endpoint('fmb')
    # fmb = VUnits.instance.hal.nodes['Mainboard']
    min_pwm = 0.0
    await _fan_pwm('int', 0)
    await _fan_pwm('ext', 0)

    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

    await fmb.SetAnalogOutput(p_1, min_pwm)
    await fmb.SetAnalogOutput(p_2, min_pwm)

    return('TCU TEC switched off.')
    
# Turns Peltiers of airventilation unit OFF
# [] tested     
async def _pmt_tec(pwm_percent = 0):
    '''
    Internal function to switch TEC Element of PMT 1 to pwm_percent
    Returns string message with status
    '''
    pwm = pwm_percent / 100
    fmb = get_node_endpoint('eef')
    p_1 = temp_elements['pmt_1']
    
    await fmb.SetAnalogOutput(p_1, pwm)
    
    return(f'PMT TEC switched to {pwm}%')

async def _scantable_heater(pwm_percent = 0):
    '''
    Internal function to switch the lower heating foil (attached to the scantable unit) to pwm_percent
    Returns status message
    '''
    pwm = pwm_percent / 100
    fmb = get_node_endpoint('fmb')
    heater = temp_elements['htl']
    await fmb.SetAnalogOutput(heater, pwm)
    return(f'Heating switched to {pwm_percent}%')
